/**
 *    Copyright (C) 2014 MongoDB Inc.
 *
 *    This program is free software: you can redistribute it and/or  modify
 *    it under the terms of the GNU Affero General Public License, version 3,
 *    as published by the Free Software Foundation.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Affero General Public License for more details.
 *
 *    You should have received a copy of the GNU Affero General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the GNU Affero General Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include <boost/thread.hpp>
#include <boost/thread/condition.hpp>
#include <vector>

#include "mongo/base/disallow_copying.h"
#include "mongo/base/status_with.h"
#include "mongo/bson/oid.h"
#include "mongo/bson/timestamp.h"
#include "mongo/db/jsobj.h"
#include "mongo/db/repl/replication_coordinator_external_state.h"
#include "mongo/db/repl/last_vote.h"
#include "mongo/util/net/hostandport.h"

namespace mongo {
namespace repl {

    class ReplicationCoordinatorExternalStateMock : public ReplicationCoordinatorExternalState {
        MONGO_DISALLOW_COPYING(ReplicationCoordinatorExternalStateMock);
    public:
        class GlobalSharedLockAcquirer;

        ReplicationCoordinatorExternalStateMock();
        virtual ~ReplicationCoordinatorExternalStateMock();
        virtual void startThreads();
        virtual void startMasterSlave(OperationContext*);
        virtual void shutdown();
        virtual void initiateOplog(OperationContext* txn);
        virtual void forwardSlaveProgress();
        virtual OID ensureMe(OperationContext*);
        virtual bool isSelf(const HostAndPort& host);
        virtual HostAndPort getClientHostAndPort(const OperationContext* txn);
        virtual StatusWith<BSONObj> loadLocalConfigDocument(OperationContext* txn);
        virtual Status storeLocalConfigDocument(OperationContext* txn, const BSONObj& config);
        virtual StatusWith<LastVote> loadLocalLastVoteDocument(OperationContext* txn);
        virtual Status storeLocalLastVoteDocument(OperationContext* txn, const LastVote& lastVote);
        virtual void setGlobalTimestamp(const Timestamp& newTime);
        virtual StatusWith<OpTime> loadLastOpTime(OperationContext* txn);
        virtual void closeConnections();
        virtual void killAllUserOperations(OperationContext* txn);
        virtual void clearShardingState();
        virtual void signalApplierToChooseNewSyncSource();
        virtual OperationContext* createOperationContext(const std::string& threadName);
        virtual void dropAllTempCollections(OperationContext* txn);

        /**
         * Adds "host" to the list of hosts that this mock will match when responding to "isSelf"
         * messages.
         */
        void addSelf(const HostAndPort& host);

        /**
         * Sets the return value for subsequent calls to loadLocalConfigDocument().
         */
        void setLocalConfigDocument(const StatusWith<BSONObj>& localConfigDocument);

        /**
         * Sets the return value for subsequent calls to loadLocalLastVoteDocument().
         */
        void setLocalLastVoteDocument(const StatusWith<LastVote>& localLastVoteDocument);

        /**
         * Sets the return value for subsequent calls to getClientHostAndPort().
         */
        void setClientHostAndPort(const HostAndPort& clientHostAndPort);

        /**
         * Sets the return value for subsequent calls to loadLastOpTimeApplied.
         */
        void setLastOpTime(const StatusWith<OpTime>& lastApplied);

        /**
         * Sets the return value for subsequent calls to storeLocalConfigDocument().
         * If "status" is Status::OK(), the subsequent calls will call the underlying funtion.
         */ 
        void setStoreLocalConfigDocumentStatus(Status status);

        /**
         * Sets whether or not subsequent calls to storeLocalConfigDocument() should hang
         * indefinitely or not based on the value of "hang".
         */
        void setStoreLocalConfigDocumentToHang(bool hang);

        /**
         * Sets the return value for subsequent calls to storeLocalLastVoteDocument().
         * If "status" is Status::OK(), the subsequent calls will call the underlying funtion.
         */ 
        void setStoreLocalLastVoteDocumentStatus(Status status);

        /**
         * Sets whether or not subsequent calls to storeLocalLastVoteDocument() should hang
         * indefinitely or not based on the value of "hang".
         */
        void setStoreLocalLastVoteDocumentToHang(bool hang);

    private:
        StatusWith<BSONObj> _localRsConfigDocument;
        StatusWith<LastVote> _localRsLastVoteDocument;
        StatusWith<OpTime>  _lastOpTime;
        std::vector<HostAndPort> _selfHosts;
        bool _canAcquireGlobalSharedLock;
        Status _storeLocalConfigDocumentStatus;
        Status _storeLocalLastVoteDocumentStatus;
        // mutex and cond var for controlling stroeLocalConfigDocument()'s hanging
        boost::mutex _shouldHangConfigMutex;
        boost::condition _shouldHangConfigCondVar;
        // mutex and cond var for controlling stroeLocalLastVoteDocument()'s hanging
        boost::mutex _shouldHangLastVoteMutex;
        boost::condition _shouldHangLastVoteCondVar;
        bool _storeLocalConfigDocumentShouldHang;
        bool _storeLocalLastVoteDocumentShouldHang;
        bool _connectionsClosed;
        HostAndPort _clientHostAndPort;
    };

} // namespace repl
} // namespace mongo
